<?php 

function getAllUsers($pdo) {
    // Prepare a SQL query to retrieve all users
    $stmt = $pdo->prepare('SELECT * FROM users');

    // Execute the query
    $stmt->execute();

    // Fetch the users as an associative array
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Initialize an empty array to store the user objects
    $userObjects = [];

    // Loop through the users and create a user object for each user
    foreach ($users as $user) {
        // Prepare a SQL query to retrieve the user's details and wallet balance
        $stmt = $pdo->prepare('SELECT * FROM user_profiles WHERE user_id = :user_id');
        $stmt2 = $pdo->prepare('SELECT SUM(balance) as wallet_balance FROM wallets WHERE user_id = :user_id');

        // Bind the user ID parameter
        $stmt->bindParam(':user_id', $user['user_id']);
        $stmt2->bindParam(':user_id', $user['user_id']);

        // Execute the queries
        $stmt->execute();
        $stmt2->execute();

        // Fetch the user's details and wallet balance as associative arrays
        $userDetails = $stmt->fetch(PDO::FETCH_ASSOC);
        $walletBalance = $stmt2->fetch(PDO::FETCH_ASSOC);

        // Merge the user, user details, and wallet balance arrays
        $userData = array_merge($user, $userDetails, $walletBalance);

        // Create a new user object and add it to the user objects array
        $userObjects[] = (object) $userData;
    }

    // Return the user objects or an empty array if there are no users
    return $userObjects ?: [];
}


//----------------------------------------------------------------
//FUNCTION TO GET A USER COMPLETE DETAILS

function getSanitizedUserIdFromUserDetailsUrl($userDetailsUrl) {
    // Extract the user ID from the URL using parse_url() and parse_str() functions
    parse_str(parse_url($userDetailsUrl, PHP_URL_QUERY), $params);
    $userId = $params['user'] ?? '';

    // Validate the user ID
    if (!is_string($userId) || empty($userId)) {
        return '';
    }

    // Sanitize the user ID
    $sanitizedUserId = filter_var($userId, FILTER_SANITIZE_SPECIAL_CHARS);

    return $sanitizedUserId;
}




function getUserDetailsByUserId($pdo, $userId) {
// Prepare a SQL query to retrieve the user's details and wallet balance
    $stmt = $pdo->prepare('
        SELECT u.*, up.*, SUM(w.balance) as wallet_balance
        FROM users u
        JOIN user_profiles up ON u.user_id = up.user_id
        LEFT JOIN wallets w ON u.user_id = w.user_id
        WHERE u.user_id = :user_id
        GROUP BY u.user_id
    ');

    // Bind th    e user ID parameter
    $stmt->bindParam(':user_id', $userId);

    // Execute the query
    $stmt->execute();

    // Fetch the user data as an associative array
    $userData = $stmt->fetch(PDO::FETCH_ASSOC);

    // If the user data exists, create a new user object and return it
    if ($userData) {
        return (object) $userData;
    }

    // If the user data doesn't exist, return null
    return null;
}


//----------------------------------------------------------------

//FUNCTION TO GET ALL TRANSACTIONS FOR THE ADMIN

function getAllTransactions($page = 1) {
    //require_once '../config/database.php';
    global $pdo;

    $limit = 25; // Number of transactions per page
    $offset = ($page - 1) * $limit; // Calculate the offset
    $currentPage = isset($_GET['page']) && is_numeric($_GET['page']) ? (int) $_GET['page'] : 1;

    $stmt = $pdo->prepare('SELECT * FROM transactions ORDER BY created_at DESC LIMIT :limit OFFSET :offset');
    $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();

    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $paginationHtml = getPaginationLinks($transactions, $currentPage);

    $transactionsPerPage = 25;
    $offset = ($currentPage - 1) * $transactionsPerPage;

    if (empty($transactions)) {
        // Call the function if there are no transactions
        noTransactionsFound();
    } else {
        $transactionHtml = '';
        foreach ($transactions as $transaction) {
            $statusIconClass = '';

            if ($transaction['transaction_status'] == 'pending') {
                $statusIconClass = 'data-state-pending';
            } elseif ($transaction['transaction_status'] == 'approved' || $transaction['transaction_status'] == 'delivered') {
                $statusIconClass = 'data-state-approved';
            } elseif ($transaction['transaction_status'] == 'cancelled') {
                $statusIconClass = 'data-state-canceled';
            } elseif ($transaction['transaction_status'] == 'progress') {
                $statusIconClass = 'data-state-progress';
            }
            $transactionHtml .= '
<tr class="data-item">
    <td class="data-col dt-tnxno">
        <div class="d-flex align-items-center">
        <div class="data-state ' . $statusIconClass . '"><span class="d-none"></span></div>
                
            <div class="fake-class"><span class="lead tnx-id">' . htmlspecialchars($transaction['transaction_description']) . '</span><span class="sub sub-date">' . htmlspecialchars(date('d-M-Y h:i A', strtotime($transaction['created_at']))) . '</span></div>
        </div>
    </td>
    <td class="data-col dt-token"><span class="lead token-amount">' . htmlspecialchars($transaction['transaction_reference']) . '</span><span class="sub sub-symbol">' . htmlspecialchars($transaction['transaction_currency']) . '</span></td>
    <td class="data-col dt-amount"><span class="lead amount-pay">' . htmlspecialchars(number_format($transaction['transaction_amount'], 2, '.', ',')) . '</span><span class="sub sub-symbol">' . htmlspecialchars($transaction['transaction_currency']) . ' </span></span></td>
    <td class="data-col dt-usd-amount"><span class="lead amount-pay">' . htmlspecialchars(number_format($transaction['transaction_total_amount'], 2, '.', ',')) . '</span><span class="sub sub-symbol">NGN </span></td>
    <td class="data-col dt-account"><span class="lead user-info">' . htmlspecialchars($transaction['transaction_status']) . '</span></td>
    <td class="data-col dt-type"><span class="dt-type-md badge badge-outline badge-success badge-md">' . htmlspecialchars(ucfirst($transaction['transaction_type'])) . '</span><span class="dt-type-sm badge badge-sq badge-outline badge-success badge-md">' . htmlspecialchars(strtoupper($transaction['transaction_type'][0])) . '</span></td>
    <td class="data-col text-right"><a href="transaction-details?tranx-ref=' . htmlspecialchars($transaction['transaction_reference']) . '" class="btn btn-light-alt btn-xs btn-icon"><em class="ti ti-eye"></em></a></td>
</tr>
';
}

return $transactionHtml;
}
}

function noTransactionsFound() {
// Code to execute if there are no transactions found
echo 'No transactions found.';
}








function getPaginationLinks($transactions, $currentPage) {
    $limit = 25; // Number of transactions per page
    $totalTransactions = count($transactions); // Total number of transactions
    $totalPages = ceil($totalTransactions / $limit); // Total number of pages
    $startPage = max(1, $currentPage - 5); // Start page for pagination links
    $endPage = min($totalPages, $startPage + 9); // End page for pagination links

    $paginationHtml = '';

    if ($currentPage > 1) {
        $paginationHtml .= '<li class="page-item"><a class="page-link" href="?page=' . ($currentPage - 1) . '">&laquo;</a></li>';
    }

    for ($i = $startPage; $i <= $endPage; $i++) {
        if ($i == $currentPage) {
            $paginationHtml .= '<li class="page-item active"><span class="page-link">' . $i . '</span></li>';
        } else {
            $paginationHtml .= '<li class="page-item"><a class="page-link" href="?page=' . $i . '">' . $i . '</a></li>';
        }
    }

    if ($currentPage < $totalPages) {
        $paginationHtml .= '<li class="page-item"><a class="page-link" href="?page=' . ($currentPage + 1) . '">&raquo;</a></li>';
    }

    return $paginationHtml;
}


//----------------------------------------------------------------

//GET ALL KYC DETAILS

// function getAllKYCApplications() {
//     //require_once '../config/database.php';
//     global $pdo;

//     // Prepare the SQL statement
//     $stmt = $pdo->prepare('SELECT * FROM kyc');

//     // Execute the SQL statement
//     $stmt->execute();

//     // Fetch all the KYC applications as an associative array
//     $kycApplications = $stmt->fetchAll(PDO::FETCH_ASSOC);

//     // Initialize the HTML output
//     $html = '<table class="data-table dt-init kyc-list">
//                         <thead>
//                             <tr class="data-item data-head">
//                                 <th class="data-col dt-user">User Id</th>
//                                 <th class="data-col dt-user">Username</th>
//                                 <th class="data-col dt-doc-type">Email</th>
//                                 <th class="data-col dt-status">Status</th>
//                                 <th class="data-col"></th>
//                             </tr>
//                         </thead>
//                         <tbody>';

//     // Loop through the KYC applications and generate the HTML output
//     foreach ($kycApplications as $kycApplication) {
//         $statusIconClass = '';

//         if ($kycApplication['status'] == 'pending') {
//             $statusIconClass = 'badge-warning';
//         } elseif ($kycApplication['status'] == 'approved') {
//             $statusIconClass = 'badge-success';
//         } elseif ($kycApplication['status'] == 'rejected') {
//             $statusIconClass = 'badge-danger';
//         }

//         $html .= '
//         <tr class="data-item">
//             <td class="data-col dt-user"><span class="lead user-name">' . htmlspecialchars($kycApplication['user_id']) . '</span></td>
//             <td class="data-col dt-user"><span class="sub user-id">' . htmlspecialchars($kycApplication['user_first_name']) . ' '. htmlspecialchars($kycApplication['user_last_name']) . '</span></td>
//             <td class="data-col dt-doc-type"><span class="sub sub-s2 sub-dtype">' . htmlspecialchars($kycApplication['user_email']) . '</span></td>
//             <td class="data-col dt-status"><span class="dt-status-md badge badge-outline ' . $statusIconClass . ' badge-md">' . htmlspecialchars($kycApplication['status']) . '</span><span class="dt-status-sm badge badge-sq badge-outline ' . $statusIconClass . ' badge-md">' . htmlspecialchars(substr($kycApplication['status'], 0, 1)) . '</span></td>
//             <td class="data-col text-right">
//                 <div class="relative d-inline-block">
//                     <a href="#" class="btn btn-light-alt btn-xs btn-icon toggle-tigger"><em class="ti ti-more-alt"></em></a>
//                     <div class="toggle-class dropdown-content dropdown-content-top-left">
//                         <ul class="dropdown-list">
//                             <li><a href="kyc-details.php?kyc_id=' . htmlspecialchars($kycApplication['id']) . '"><em class="ti ti-eye"></em> View Details</a></li>
//                             <li><a href="#"><em class="ti ti-check"></em> Approve</a></li>
//                             <li><a href="#"><em class="ti ti-na"></em> Cancel</a></li>
//                             <li><a href="#"><em class="ti ti-trash"></em> Delete</a></li>
//                         </ul>
//                     </div>
//                 </div>
//             </td>
//         </tr>
//         ';
//     }

    function getAllKYCApplications() {
        //require_once '../config/database.php';
        global $pdo;
    
        // Prepare the SQL statement
        $stmt = $pdo->prepare('SELECT * FROM kyc');
    
        // Execute the SQL statement
        $stmt->execute();
    
        // Fetch all the KYC applications as an associative array
        $kycApplications = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
        // Initialize the HTML output
        $html = '<table class="data-table dt-init kyc-list">
                            <thead>
                                <tr class="data-item data-head">
                                    <th class="data-col dt-user">User Id</th>
                                    <th class="data-col dt-user">Username</th>
                                    <th class="data-col dt-doc-type">Email</th>
                                    <th class="data-col dt-status">Status</th>
                                    <th class="data-col"></th>
                                </tr>
                            </thead>
                            <tbody>';
    
        // Loop through the KYC applications and generate the HTML output
        foreach ($kycApplications as $kycApplication) {
            $statusIconClass = '';
    
            if ($kycApplication['status'] == 'pending') {
                $statusIconClass = 'badge-warning';
            } elseif ($kycApplication['status'] == 'approved') {
                $statusIconClass = 'badge-success';
            } elseif ($kycApplication['status'] == 'rejected') {
                $statusIconClass = 'badge-danger';
            }
    
            $html .= '
            <tr class="data-item">
                <td class="data-col dt-user"><span class="lead user-name">' . htmlspecialchars($kycApplication['user_id']) . '</span></td>
                <td class="data-col dt-user"><span class="sub user-id">' . htmlspecialchars($kycApplication['user_first_name']) . ' ' . htmlspecialchars($kycApplication['user_last_name']) . '</span></td>
                <td class="data-col dt-doc-type"><span class="sub sub-s2 sub-dtype">' . htmlspecialchars($kycApplication['user_email']) . '</span></td>
                <td class="data-col dt-status"><span class="dt-status-md badge badge-outline ' . $statusIconClass . ' badge-md">' . htmlspecialchars($kycApplication['status']) . '</span><span class="dt-status-sm badge badge-sq badge-outline ' . $statusIconClass . ' badge-md">' . htmlspecialchars(substr($kycApplication['status'], 0, 1)) . '</span></td>
                <td class="data-col text-right">
                    <div class="relative d-inline-block">
                        <a href="#" class="btn btn-light-alt btn-xs btn-icon toggle-tigger"><em class="ti ti-more-alt"></em></a>
                        <div class="toggle-class dropdown-content dropdown-content-top-left">
                            <ul class="dropdown-list">
                                <li><a href="kyc-details?kyc_id=' . htmlspecialchars($kycApplication['id']) . '"><em class="ti ti-eye"></em> View Details</a></li>
                                <li><a href="#"><em class="ti ti-check"></em> Approve</a></li>
                                <li><a href="#"><em class="ti ti-na"></em> Cancel</a></li>
                                <li><a href="#"><em class="ti ti-trash"></em> Delete</a></li>
                            </ul>
                        </div>
                    </div>
                </td>
            </tr>
            ';
        }
    
        // Close the HTML output
        $html .= '
            </tbody>
        </table>';
    
        // Return the HTML output
         
    return $html;
}



//------------------------------------------------------------------------------------------------------
//FUNCTION TO GET USER KYC DETAILS 

function get_kyc_id_from_url($url) {
    $parsed_url = parse_url($url);
    if (isset($parsed_url['query'])) {
        parse_str($parsed_url['query'], $query_params);
        if (isset($query_params['kyc_id'])) {
            $kyc_id = $query_params['kyc_id'];
            if (filter_var($kyc_id, FILTER_VALIDATE_INT)) {
                return $kyc_id;
            }
        }
    }
    return null;
}